-- glsl shaders
-- #version 120

-- format is { COMMON_HEADER, VERTEX_SHADER, FRAGMENT_SHADER }

-- implicit parameters:
-- attribute vec4 Position;
-- uniform mat4 Transform;

{
    -- spaceship hulls
    ShaderIridescent = {
        "varying vec4 DestinationColor;"
        ,
        "attribute vec4 SourceColor0;
        attribute vec4 SourceColor1;
        attribute float TimeA;

        // Include this in order to use snoise
        #include 'noise3D.glsl'

        void main(void) {
            gl_Position = Transform * Position;
            // snoise is normally used to draw the wormhole
            float val = snoise(vec3(0, Time, TimeA));

            // Sprinkle in some more iridescence and normalize overly dark and light blocks
            DestinationColor = vec4(0.2) + mix(SourceColor0, SourceColor1, val)
                * vec4(normalize(vec3(sin(Time*2), sin(Time*2+TimeA), sin(Time*2 + 2*TimeA)))*0.035 + vec3(0.66), 0.75);
        }"
        ,
        "
        void main(void) {
              gl_FragColor = DestinationColor;
        }"
    },

    -- draws projectiles, shields, lasers, etc
    ShaderColorLuma = {
        "varying vec4 DestinationColor;"
        ,
        "attribute vec4 SourceColor;
        attribute float Luma;
        void main(void) {
            DestinationColor = Luma * SourceColor;
            gl_Position = Transform * Position;
        }"
        ,
        "void main(void) {
            gl_FragColor = DestinationColor;
            gl_FragColor.a -= 0.01;
        }" 
    },

    -- draws stars
    ShaderStars = {
        "varying vec4 DestinationColor;"
        ,
        "attribute vec4 SourceColor;
        attribute float Size;
        uniform float ToPixels;
        uniform float EyeZ;

        void main(void) {
            DestinationColor = SourceColor;
            gl_PointSize = ToPixels * Size / (EyeZ - Position.z);
            gl_Position = Transform * Position;
        }"
        ,
        "void main(void) {
            vec2 coord = 2.0 * (gl_PointCoord - 0.5);
            float val = (length(coord) + abs(coord.x) + abs(coord.y)) / 2.0;
            float alpha = min(1.0 - val, 1.0);
            if (alpha <= 0.0)
                discard;

            gl_FragColor = DestinationColor * vec4(1.5, 1.5, 1.5, alpha * 1.3);
        }"
    },

    ShaderWorley = {
      ""
      ,
      "void main(void) {
          gl_Position = Transform * Position;
      }"
      ,
      "uniform vec2  points[POINTS];
      uniform vec3  color;
      uniform float scale;
      uniform sampler2D dither;
      void main(void) {
          float minv = 100000000000000.0;
          for (int i=0; i<POINTS; i++)
          {
               vec2 d = points[i] - gl_FragCoord.xy;
               float v = d.x * d.x + d.y * d.y;
               minv = min(minv, v);
          }
          float myv = sqrt(minv) * scale;
          if (myv > 1.0) {
              discard;
          }
          float alpha = 1.0 - myv;
          gl_FragColor = vec4(alpha * color.rgb, alpha);

          #if DITHER
              //alpha += texture2D(dither, gl_FragCoord.xy / 8.0).r / 4.0 - (1.0 / 128.0);
              gl_FragColor += vec4(texture2D(dither, gl_FragCoord.xy / 8.0).r / 16.0 - (1.0 / 64.0));
          #endif

          gl_FragColor *= 0.75;

      }",
    },

    -- passthrough texture shader - used in many places for drawing render targets
    ShaderTexture = {
        "varying vec2 DestTexCoord;
        varying vec4 DestPosition;
        varying vec4 DestColor;\n"
        ,
        "attribute vec2 SourceTexCoord;
        uniform vec4 SourceColor;
        void main(void) {
            DestTexCoord = SourceTexCoord;
            DestColor    = SourceColor;
            DestPosition = Position;
            gl_Position  = Transform * Position;
        }"
        ,

        "uniform sampler2D texture1;
        void main(void) {

            vec2 direction = DestTexCoord - vec2(0.5);

            // Increase effect towards the middle but not at the edges where there is text
            direction = vec2((-0.5 * direction.x / (0.001 + abs(direction.x)) + direction.x) * abs(-0.5 * direction.y / (0.001 + abs(direction.y)) + direction.y), 0);

            // Remove screen-tearing effect
            direction.x = direction.x * (DestTexCoord.x - 0.5);

            // You can also use: vec2 direction = vec2(0.001, -0.0002);

            vec4 color = texture2D(texture1, DestTexCoord);

            if (DestPosition.x > 1 && DestPosition.y > 1)
            {
                color.a *= 1.05;
                if (length(DestColor.rgb) < 0.9 && max(DestColor.r, max(DestColor.g, DestColor.b)) - min (DestColor.r, min(DestColor.g, DestColor.b)) > 0.1) color.a *= 1.75 / length(DestColor);
                gl_FragColor = DestColor * color;
            }
            else
            {
                gl_FragColor = DestColor * vec4(

                    color.r*0.2 + 0.8*texture2D(texture1, DestTexCoord - direction * - 0.02).r,
                    color.g,
                    color.b*0.2 + 0.8*texture2D(texture1, DestTexCoord - direction * 0.03).b,

                    // Replace the following line with 1 to experience CIA Mode
                    color.a
                );
            }
        }"
    },

    -- HDR postprocessing. makes really bright areas (i.e. with a lot of particles) more white to make them look brighter
    ShaderTonemap = {
      "varying vec2 DestTexCoord;"
      ,
      "attribute vec2 SourceTexCoord;
      void main(void) {
          DestTexCoord = SourceTexCoord;
          gl_Position  = Transform * Position;
      }"
      ,
      "uniform sampler2D texture1;
      uniform sampler2D dithertex;
      void main(void) {
          vec2 texCoord = DestTexCoord;
          vec4 color = texture2D(texture1, texCoord);

          if (color.rgb != vec3(0.0))
          {
              // Prevent already-white colors from overpowering the screen
              float mx = max(1.5 * color.r - color.g - color.b,  max(1.5 * color.g - color.b - color.r, 1.5 * color.b - color.r - color.g));
              float mx2 = max(color.r + color.g - 3 * color.b,  max(color.g + color.b - 3 * color.r, color.b + color.r - 3 * color.g)) / 2;
              mx = max(mx, mx2);

              if (mx > 0.05) {
                color.rgb += (normalize(color.rgb) * 0.5  + color.rgb * 0.3 + color.a * (color.a + 0.2) * vec3(max(-0.75, 0.0 + mx)) ) * max(0.05, min(2, 1 + log(mx))) * 0.7;
              } else {
                // color.rgb *= max(0.5, 1 - 0.2 * (0.05 - mx));
              }

              // Reduce glare from multiple command modules
              if (color.a < 0.175) color.rgb *= 1 / (1 + 0.075 * pow(log( 1 + (color.r + color.g + color.b) * (0.175 - color.a)), 5));
              
              #if DITHER
                float ditherv = texture2D(dithertex, gl_FragCoord.xy / 8.0).r / 128.0 - (1.0 / 128.0);
                color += vec4(ditherv);
              #endif
          }

          // Adjust based on particle brightness preference
          gl_FragColor = vec4(color.rgb * 0.6, color.a * 0.95);

      }"
   },

    -- used for particles (GL_POINTS)
    ShaderParticlePoints = {
        "varying vec4 DestinationColor;
        varying float Sides;"
        ,
        "attribute vec2  Offset;
        attribute float StartTime;
        attribute float EndTime;
        attribute vec3  Velocity;
        attribute vec4  Color;
        uniform   float CurrentTime;
        uniform   float ToPixels;
        void main(void) {

            float deltaT = (CurrentTime - StartTime);
            float v = deltaT / (EndTime - StartTime);

            float zoffset = 8;

            // Shrink over time
            float size = max(0, 0.05 + Offset.x * min(3, 6 - v)/2.4);

            if (ToPixels * size < 0.075) {
                gl_Position = vec4(0.0, 0.0, -99999999.0, 1);
                return;
            }

            // Adhere more to the trail
            vec3 velocity = pow(0.5, deltaT) * Velocity / 5;
            if (EndTime - StartTime > 1.5) velocity = pow(0.6, deltaT) * Velocity / 4;

            vec3 position = Position.xyz + deltaT * velocity;

            // Secret Mode: * (25 + 25 * vec4(normalize(velocity)/8 + vec3(0.3), Color.a))
            DestinationColor = Color + vec4(sin(deltaT*2), sin(deltaT*2+CurrentTime), sin(deltaT*2 + 4*CurrentTime), 1) * 0.005;

            if (EndTime - StartTime > 1.5) {

                if (max(Color.r, max(Color.g, Color.b)) < 0.25 && Color.a > 0.75)
                {
                    // smoke
                    zoffset = -1;
                    DestinationColor *= max(vec4(0), vec4(vec3(0.2 - 0.2 * v), 0.15 - 0.15 * v));
                    size = size * 1.075 + 1;
                }
                else
                {
                    // long-distance thruster trails

                    size *= 1.5;
                    DestinationColor *= max(0.8, 20 / (2 + Offset.x) );
                    DestinationColor *= vec4(max(0.0, (2 - (EndTime - StartTime)/2 - 0.75 * v)/6) );

                    // eliminate phantom trails
                    if (size < 0.4) {
                        gl_Position = vec4(0.0, 0.0, -99999999.0, 1);
                        return;
                    }
                }
            }
            else
            {
                DestinationColor.rgba *= max(0.5, 10 / (3 + size) );

                if (Offset.x > 30)
                {
                    // nukes
                    zoffset = -1;
                    size = Offset.x * 2;
                    DestinationColor *= 0.9 * vec4(vec3(max(0,(0.15 - deltaT/2)/(0.3 + deltaT/2))) + min(1, 0.5 + Offset.x/20) * vec3(max(0, min(1, 1 - 1.5 * v * v))), max(0, 0.95 - 0.75 * v));
                }
                else
                {
                    // general particles

                    //size *= (0.75 + 0.5 * max(Color.r, max(Color.g, Color.b)));

                    //position = Position.xyz + (2 + deltaT/2) * velocity;

                    DestinationColor *= max(0.0, min(1.6, (2 - (EndTime - StartTime)/8 - 0.5 * v)/5));

					// healing trails

					if (Offset.x < 9.129 && Offset.x > 9.1285)
					{
						zoffset = -1;
						DestinationColor *= vec4(0.3, 0.3, 0.3, 0) * max(0, EndTime - CurrentTime);
						position = Position.xyz + deltaT * velocity * 5;
						size *= 2.5;
					}
                }
            }

            /*
            if (EndTime - StartTime < 0.5) DestinationColor = vec4(0,1,0,1);
            else if (EndTime - StartTime < 1) DestinationColor = vec4(1,0,0,1);
            else DestinationColor = vec4(0,0,1,1);
            */

            Sides = Offset.y;
            gl_PointSize = ToPixels * size;
            gl_Position = Transform * vec4(position, 1);

            gl_Position.z += zoffset;
        }"
        ,
        "
        float length2(vec2 x) { return dot(x, x); }
        float mlength(vec2 x) { return abs(x.x) + abs(x.y); }
        float squared(float x) { return x * x; }

        void main(void) {
            //float val = 1.0 - squared(2.0 * length(gl_PointCoord - 0.5));
            float val = 1.0 - 4.0 * length2(gl_PointCoord - 0.5);
            //float val = 1.0 - 2.0 * mlength(gl_PointCoord - 0.5);
            //float val = 1.0 - 2.0 * length(gl_PointCoord - 0.5);
            if (val <= 0.0)
                discard;
            //val = max(0.0, val);
            if (Sides > 0.0) {
                gl_FragColor = DestinationColor;
            } else {
                gl_FragColor = DestinationColor * val;
            }
        }"
    }

   -- full screen blurs.
   ShaderBlur = {
      "varying vec2 DestTexCoord;"
      ,
      "attribute vec2 SourceTexCoord;
       void main(void) {
           DestTexCoord = SourceTexCoord;
           gl_Position  = Transform * Position;
       }"
      ,
      -- BLUR expands to the sum of a number of texture2d calls depending on the blur radius
      "uniform sampler2D source;
       uniform vec2 offsets[BLUR_SIZE];
       void main() {
           // Adjust based on background afterimage brightness preference
           gl_FragColor = vec4(0.9,0.9,0.9,0.95) * BLUR(source, DestTexCoord, offsets);
       }"
   }
}
